C      *****************************************************************
C      * Subroutine InDat                                              *
C      * Reads in model data from the file name passed in FName.  The  *
C      * file is in the form of a variable name/descriptor followed by *
C      * the input data for that variable on the next line.            *
C      *                                                               *
C      * Variables:                                                    *
C      * ActCorr   -Output- A vector, of size NSpecies, of flags to    *
C      *                    indicate how activity corrections will be  *
C      *                    calculated.                                *
C      *                    1 = None (concentration = activity),       *
C      *                    2 = Davies,                                *
C      *                    3 = Debye-Huckle (not included but easy to *
C      *                        add if desired),                       *
C      *                    4 = Vanselow exchange,                     *
C      *                    5 = Gaines-Thomas exchange.                *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * C         -Local - Component index.  Unitless.                *
C      * Ca_in_OM  -Output- The amount of calcium in organic matter.   *
C      *                    Units are g Ca.                            *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * CAmt      -Output- A matrix, of size 4 x NLayers, of the      *
C      *                    amount of carbon in cellulose, lignin,     *
C      *                    extractives and humus for each layer.      *
C      *                    Units are g C/m**2.                        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Carb      -Input - The array index for carbon.  Unitless.     *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * CE        -Input - The array index for cellulose to           *
C      *                    extractives.  Unitless.                    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Cell      -Input - The array index for cellulose.  Unitless.  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Comment   -Output- The description of the input file.         *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * CtoN      -Output- A vector, of size four, of the carbon to   *
C      *                    nitrogen ratio for cellulose, extractives, *
C      *                    lignin, and humus.  Unitless.              *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * EL        -Input - The array index for extractives to lignin. *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * EqCnst    -Output- A matrix, of size NSpecies by NLayers, of  *
C      *                    the equilibrium constants used for species *
C      *                    formation.  The units depend on            *
C      *                    stochiometric coefficients (SC) and the    *
C      *                    units of species concentrations (SpConc)   *
C      *                    (mol/L).                                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * Extr      -Input - The array index for extractives.  Unitless.*
C      *                    (Common block VDecomp, file VModel.f)      *
C      * FName     -Input - The name of the file to be read in by      *
C      *                    the routine InDat. Input from TREGRO, or   *
C      *                    read into the model by the routine InParam.*
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * Fracts    -Output- A three dimesional matrix, of size 3 by 2  *
C      *                    by NLayers, which holds the fraction of    *
C      *                    carbon and nitrogen in the different root  *
C      *                    parts for each layer, as follows:          *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    ~                 CARBON    NITROGEN    ~  *
C      *                    ~               (fraction) (fraction)   ~  *
C      *                    ~    CELLULOSE      ##        ##        ~  *
C      *                    ~    EXTRACTIVES    ##        ##        ~  *
C      *                    ~    LIGNIN         ##        ##        ~  *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    Update 3/93 => no more fraction for        *
C      *                    nitrogen.  Doesn't take up much room so am *
C      *                    leaving it in for now.  Unitless.          *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * GoAhead   -Output- Indicates whether any errors were found.   *
C      *                    (GoAhead=1, no errors found; GoAhead=0,    *
C      *                    errors found.)  Unitless.                  *
C      * HalfSat   -Output- Decomposition N half saturation constant.  *
C      *                    Units are g N/m**2.                        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * HE        -Input - The array index for humus to extractives.  *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * HOrder    -Input - A vector, of size NMnrls, of reaction      *
C      *                    orders with respect to hydrogen ion        *
C      *                    concentration, for each mineral.  Unitless.*
C      *                    (Common block VModel, file VModel.f)       *
C      * HoldAC    -Output- A vector, of size NSpecies, that holds the *
C      *                    "named" input value for the activity       *
C      *                    correction coefficient.  This "name"       *
C      *                    gets converted to a number for use in the  *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      * HoldSType -Output- A vector, of size NSpecies, that holds the *
C      *                    "named" input value for SType.  This "name"*
C      *                    gets converted to a number for use in the  *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      * Humu      -Input - The array index for humus.  Unitless.      *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * ILC       -Output- A vector, of size 2, of ILC shielding      *
C      *                    parameters for cellulose and extractives.  *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * InLay     -Local - The number of layers read into this        *
C      *                    routine from the input file.  Unitless.    *
C      * K_in_OM   -Output- The amount of potassium in organic matter. *
C      *                    Units are g K.                             *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * KT        -Output- A matrix, of size NComp by NLayer, of the  *
C      *                    known total amount for each component, for *
C      *                    each layer.  The total includes the amount *
C      *                    of the component as a free component and as*
C      *                    part of all dependent species.  Units are  *
C      *                    mol.                                       *
C      *                    (Common block VModel, file VModel.f)       *
C      * L         -Local - Layer number index.  Unitless.             *
C      * Labels    -Output- A vector, of size NSpecies, of the name for*
C      *                    each species.  Unitless.                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * LE        -Input - The array index for lignin to extractives. *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LfFracts  -Output- A three dimesional matrix, of size 3 by 2  *
C      *                    by NLayers, which holds the fraction of    *
C      *                    carbon and nitrogen in the senesced leaves,*
C      *                    as follows:                                *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    ~                 CARBON    NITROGEN    ~  *
C      *                    ~               (fraction) (fraction)   ~  *
C      *                    ~    CELLULOSE      ##        ##        ~  *
C      *                    ~    EXTRACTIVES    ##        ##        ~  *
C      *                    ~    LIGNIN         ##        ##        ~  *
C      *                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
C      *                    Update 3/93 => no more fraction for        *
C      *                    nitrogen.  Doesn't take up much room so am *
C      *                    leaving it in for now.  Unitless.          *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LH        -Input - The array index for lignin to humus.       *
C      *                    Unitless.                                  *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Lign      -Input - The array index for lignin.  Unitless.     *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LIn       -Local - Long array index, i.e. for arrays of       *
C      *                    length NSpecies (components are followed   *
C      *                    by dependent species).  Unitless.          *
C      * MaxNumLayers                                                  *
C      *           -Local - The maximal number of layers used by       *
C      *                    YASEQL stack.  Used for reading in full    *
C      *                    arrays.  Unitless.                         *
C      * MaxSLTemp -Output- The maximal soil temperature for           *
C      *                    decomposition.  Units are deg C.           *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * MinFrac   -Output- A matrix, of size NMnrls by NLayers, of    *
C      *                    the amount of each mineral that is in each *
C      *                    layer as a fraction of soil weight. Units  *
C      *                    are g/g and the values range between zero  *
C      *                    and one.                                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * MinLab    -Output- A vector, of size NMnrls, of the name for  *
C      *                    each mineral.  Unitless.                   *
C      *                    (Common block VModel, file VModel.f)       *
C      * MinMW     -Output- A vector, of size NMnrls, of formula       *
C      *                    weights for each mineral.  Units are       *
C      *                    g/mol.                                     *
C      *                    (Common block VModel, file VModel.f)       *
C      * MinSC     -Output- An array, of size NMnrls by NComp, of the  *
C      *                    stoichiometric coefficient of each         *
C      *                    component for each mineral.  Unitless.     *
C      *                    (Common block VModel, file VModel.f)       *
C      * Mg_in_OM  -Output- The amount of magnesium in organic matter. *
C      *                    Units are g Mg.                            *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * MOrder    -Input - A vector, of size NMnrls, of reaction      *
C      *                    orders with respect to MinFrac, for each   *
C      *                    mineral.  Unitless.                        *
C      *                    (Common block VModel, file VModel.f)       *
C      * Msg       -Local - An array of output messages.  Unitless.    *
C      * NAmt      -Output- A matrix, of size 4 by NLayers, of the     *
C      *                    amount of nitrogen in  cellulose, lignin,  *
C      *                    extractives and humus for each layer.      *
C      *                    Units are g N/m**2.                        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * NComp     -Output- The number of components in the system.    *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NLayersSize                                                   *
C      *           -Input - The max number of layers, used to size     *
C      *                    arrays.  Unitless.                         *
C      *                    (File ArraySizes.Inc)                      *
C      * NCompSize -Input - The max number of components, used to size *
C      *                    arrays.  Unitless.                         *
C      *                    (file ArraySizes.Inc)                      *
C      * NDepSp    -Output- The number of dependent species in the     *
C      *                    system.  It is equal to "NSpecies - NComp".*
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NLayers   -Input - The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * NMnrls    -Output- The number of minerals in the system.      *
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * NPhases   -Output- The number of phases in the system.        *
C      *                    Unitless.                                  *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * NSpecies  -Output- The number of species in the system (i.e.  *
C      *                    the number of components plus the number   *
C      *                    of dependent species).  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *
C      * OptSLTemp -Output- The optimal soil temperature.  Units are   *
C      *                    deg C.                                     *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * OutInfo   -Output- Indicates whether or not detailed          *
C      *                    information is output.  Unitless.          *
C      *                    (Common block VModel, file VModel.f)       *              
C      * P_in_OM   -Output- The amount of phosphorus in organic        *
C      *                    matter.  Units are g P.                    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * PhaseNames-Output- A vector, of size NPhases, of the name for *
C      *                    each phase.  Unitless.                     *
C      *                    (Common block VSolidPhase, file VModel.f)  *
C      * QCurve    -Output- For decomposition, the q curvature         *
C      *                    parameter, calculated from the optimal and *
C      *                    maximal temperatures and the ratio between *
C      *                    the Q fcn when the temperature is optimal  *
C      *                    and when it is zero.  Unitless.            *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * S         -Local - Dependent species index.  Unitless.        *
C      * SC        -Output- A matrix, of size NDepSp by NComp, of the  *
C      *                    stoichiometric coefficient of each         *
C      *                    component for each species.  Unitless.     *
C      *                    (Common block VModel, file VModel.f)       *
C      * S_in_OM   -Output- The amount of sulfur in organic matter.    *
C      *                    Units are g S.                             *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * SType     -Output- A vector, of size NSpecies, of each        *
C      *                    species's type.  Unitless.                 *
C      *                    (Common block VModel, file VModel.f)       *
C      * EqCnst    -Local - Holds the entire EqCnst matrix as it is    *
C      *                    read in from data file.  Same units as the *
C      *                    EqCnst matrix.                             *
C      * TempLabels-Local - Holds the entire labels vector as it is    *
C      *                    read in from data file.  Unitless.         *
C      * TempSC    -Local - A temporary matrix, of size NDepSp by      *
C      *                    NComp, of the stoichiometric coefficient of*
C      *                    each component for each species.  Same     *
C      *                    units as the SC matrix.                    *
C      * TempSType -Local - Holds the entire SType vector as it is read*
C      *                    in from data file.  Same units as the      *
C      *                    SType vector.                              *
C      * TempZ     -Local - Holds the entire Z (charge) vector as it is*
C      *                    in from data file.  Same units as the      *
C      *                    Z (charge) vector.                         *
C      * Toss      -Local - Holds unused strings read in from input    *
C      *                    file.  Unitless.                           *
C      * TranEffic -Output- A vector, of size 5, of transformation     *
C      *                    efficiencies from                          *
C      *                    cellulose and extractives for all          *
C      *                    cellulose and extractives (g extr/g cell), *
C      *                    extractives and cellulose for all          *
C      *                    extractives and cellulose (g cell/g extr), *
C      *                    humus and extractives for all              *
C      *                    humus and extractives (g extr/g humu),     *
C      *                    lignin and extractives for all             *
C      *                    lignin and extractives (g extr/g lign),    *
C      *                    and lignin and humus for all               *
C      *                    and lignin and humus (g humu/g lign).      *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * TranRate  -Output- A vector, of size 5, of transformation     *
C      *                    rates from cellulose and extractives for   *
C      *                    all cellulose and extractives,             *
C      *                    extractives and cellulose, for all         *
C      *                    extractives and cellulose, humus and       *
C      *                    extractives, for all humus and extractives,*
C      *                    lignin and extractives, for all lignin and *
C      *                    extractives, and lignin and humus for all  *
C      *                    lignin and humus.  Units are 1/timestep.   *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * UseDecomp -Output- Indicates whether decomposition is to be   *
C      *                    run (TRUE) or not (FALSE).  Unitless.      *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * WRate     -Input - A vector, of size NComp, of rate constants *
C      *                    for each mineral.  Value and units must be *
C      *                    consistent with values of HOrder and       *
C      *                    MOrder. For example: WRate is the fraction *
C      *                    of mineral weathered per time step when    *
C      *                    MOrder=1 and HOrder=0.                     *
C      *                    (Common block VModel, file VModel.f)       *
C      * YASOut    -Input - The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      * Z         -Output- A vector, of size NComp, of unspeciated    *
C      *                    charge for each component.  Units are      *
C      *                    equivalent charge.                         *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
	      SUBROUTINE INDAT(GOAHEAD)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							CHARACTER*8 SMLTOS
       CHARACTER*12 COMPACTCORR(NCOMPSIZE)
       CHARACTER*12 TEMPLABELS(NSPECIESSIZE)
       CHARACTER*12 TEMPSTYPE(NSPECIESSIZE)
       CHARACTER*12 SPECACTCORR(NSPECIESSIZE - NCOMPSIZE)
							CHARACTER*100 MSG(5), TOSS 
							INTEGER C, GOAHEAD, TEMPZ(NSPECIESSIZE)
       INTEGER INLAY, L, LIN
       INTEGER MAXNUMLAYERS, S
       REAL*8 UNUSED_REAL, TEMPSC(NSPECIESSIZE, NCOMPSIZE)
       REAL*8 TEMPEQCNST(NSPECIESSIZE, NLAYERSSIZE)
       PARAMETER (MAXNUMLAYERS = 10)
							
C      *--------------------------------------*
C      * Initialize go ahead flag to all OK.  *
C      *--------------------------------------*
       GOAHEAD = 1
	   
C      *---------------*
C      * Open the file.*
C      *---------------*
	      OPEN(UNIT=10, FILE=FNAME)

C      *---------------------------------------------*
C      * Read in the file version number...not used. *
C      *---------------------------------------------*
       READ(10,*) TOSS
       READ(10,*) TOSS

C      *--------------------------------------------------*
C      * Read in the label, then the number of components.*
C      *--------------------------------------------------*
       READ(10,*) TOSS
	      READ(10,*) NCOMP

C      *---------------------*
C      * Read in all labels. *
C      *---------------------*
       DO 100 C = 1, 55
          READ(10,*) TOSS
          READ(10,*) TEMPLABELS(C)
  100  CONTINUE

C      *----------------------*
C      * Read in all charges. *
C      *----------------------*
       DO 110 C = 1, 55
          READ(10,*) TOSS
          READ(10,*) TEMPZ(C)
  110  CONTINUE

C      *---------------------*
C      * Read in all STypes. *
C      *---------------------*
       DO 120 C = 1, 55
          READ(10,*) TOSS
          READ(10,*) TEMPSTYPE(C)
  120  CONTINUE

C      *----------------------------------------------------------------*
C      * Read in the label then the number of layers in the input file. *
C      * If this number of layers is not the same as the number passed  *
C      * to the model then output a message indicating that layer       *
C      * numbers don't match and set a flag to alert calling routine and*
C      * do not continue processing.  Otherwise continue processing.    *
C      *----------------------------------------------------------------*
       READ(10,*) TOSS
	      READ(10,*) INLAY

	      IF (INLAY.NE.NLAYERS) THEN
          MSG(1) = ' Error-->number of layers in input file is not'
		        MSG(2) = '         equal to # of layers passed to model'
		        GOAHEAD = 0
          GOTO 9999
	      ENDIF

C      *--------------------------------------------------------------*
C      * Read in the label then the component's total concentration.  *
C      *--------------------------------------------------------------*
       DO 130 C=1,15
          DO 135 L = 1, 10
             READ(10,*) TOSS
             READ(10,*) KT(C,L)
  135     CONTINUE
  130  CONTINUE
  
C      *--------------------------------------------------------------*
C      * Read in the label, then the number of dependent species.     *
C      * Calculate the number of species in the system.               *
C      *--------------------------------------------------------------*
       READ(10,*) TOSS
	      READ(10,*) NDEPSP
	      NSPECIES = NCOMP + NDEPSP
	   
C      *--------------------------------------------------------------*
C      * Read in the stoichiometric coefficients for each component.  *     
C      *--------------------------------------------------------------*
	      DO 140 S=1, 55
          DO 145 C = 1, 15
             READ(10,*) TOSS
	            READ(10,*) TEMPSC(S,C)
  145     CONTINUE
  140  CONTINUE

C      *-----------------------------------------------------------------*
C      * Fill in the top of the SC matrix (species that are components). *
C      *-----------------------------------------------------------------*
       DO 150 S=1, NCOMP
          DO 160 C=1,NCOMP
             IF (S.EQ.C) THEN
                SC(S,C) = 1
             ELSE
                SC(S,C) = 0
             ENDIF
 160       CONTINUE
 150   CONTINUE

C      *-------------------------------------------------------*
C      * Fill in the bottom of the SC matrix (species that are *
C      * dependent species).  The L index is the start of the  *
C      * dependent species in the matrix.                      *
C      *-------------------------------------------------------*
       L = 16
       DO 165 S = NCOMP+1, NSPECIES
          DO 167 C = 1, NCOMP
             SC(S,C) = TEMPSC(L,C)
  167     CONTINUE
          L = L+1
  165  CONTINUE

C      *--------------------------------------------------------------*
C      * Read in the label and the equilibrium constant for each      *
C      * component (not used) and each dependent species...then       *
C      * reorganize the matrix.  The equilibrium const. is calculated *
C      * by taking the antilog of the input value.                    *     
C      *--------------------------------------------------------------*
	      DO 170 S=1,55
          DO 175 L=1,10
             READ(10,*) TOSS
	            READ(10,*) TEMPEQCNST(S,L)
	            TEMPEQCNST(S,L) = 10**TEMPEQCNST(S,L)
  175     CONTINUE
  170  CONTINUE

C      *-------------------------------*
C      * Fill in the top of the array. *
C      *-------------------------------*
       DO 180 S=1,NCOMP
          DO 185 L=1,NLAYERS
             EQCNST(S,L) = 1
 185      CONTINUE
 180   CONTINUE

C      *------------------------------------------------------------*
C      * Fill in the bottom of the array.  The C index is where the *
C      * dependent species start.                                   *
C      *------------------------------------------------------------*
       C = 16
       DO 187 S = NCOMP+1, NCOMP+NDEPSP
          DO 188 L = 1, NLAYERS
             EQCNST(S,L) = TEMPEQCNST(C,L)
  188     CONTINUE
          C = C + 1
  187  CONTINUE

C      *--------------------------------------------------*
C      * Read in the label, then the number of minerals   *
C      *--------------------------------------------------*
       READ(10,*) TOSS
	      READ(10,*) NMNRLS
	 
C      *-----------------------------*
C      * Read in the mineral labels. *     
C      *-----------------------------*
	      DO 190 S=1,15
          READ(10,*) TOSS
	         READ(10,*) MINLAB(S)
  190  CONTINUE

C      *-------------------------------------------------*
C      * Read the mineral stiochiometric coefficents for *
C      * in each mineral and component.                  *
C      *-------------------------------------------------*
	      DO 200 S=1,15
          DO 205 C=1,15
             READ(10,*) TOSS
	            READ(10,*) MINSC(S,C)
  205     CONTINUE
  200  CONTINUE

C      *-----------------------------------------------------*
C      * Read the fraction (between 0 and 1) of each mineral *
C      * in each layer.                                      *
C      *-----------------------------------------------------*
	      DO 210 S=1,15
          DO 215 L=1,10
             READ(10,*) TOSS
	            READ(10,*) MINFRAC(S,L)
  215     CONTINUE
  210  CONTINUE

C      *------------------------------------------------*
C      * Read in the weathering rate for each mineral.  *
C      *------------------------------------------------*
       DO 220 S=1,15
          READ(10,*) TOSS
	         READ(10,*) WRATE(S)
  220  CONTINUE

C      *------------------------------------------------*
C      * Read in the molecular weight for each mineral. *
C      *------------------------------------------------*
       DO 230 S=1,15
          READ(10,*) TOSS
	         READ(10,*) MINMW(S)
  230  CONTINUE

C      *--------------------------------------*
C      * Read in the HOrder for each mineral. *
C      *--------------------------------------*
       DO 240 S=1,15
          READ(10,*) TOSS
	         READ(10,*) HORDER(S)
  240  CONTINUE

C      *--------------------------------------*
C      * Read in the MOrder for each mineral. *
C      *--------------------------------------*
       DO 250 S=1,15
          READ(10,*) TOSS
	         READ(10,*) MORDER(S)
  250  CONTINUE

C      *----------------------------------------------------*
C      * Read throughfall concentration for each component. *
C      *----------------------------------------------------*
	      DO 260 C=1, 15
          READ(10,*) TOSS
	         READ(10,*) TFCONC(C)
  260  CONTINUE			 

C      *-------------------------------------------------------*
C      * Read in an unused variable.                           *
C      *-------------------------------------------------------*
       READ(10,*) TOSS
       READ(10,*) UNUSED_REAL

C      *-------------------------------------------------------*
C      * Read in the variable which indicates whether          *
C      * decomposition will be run or not.                     *
C      *-------------------------------------------------------*
       READ(10,*) TOSS
       READ(10,*) USEDECOMP

C      *---------------------------------------------------------*
C      * Read in the initial amounts for cellulose, extractives, *
C      * lignin and humus for each layer.                        *
C      *---------------------------------------------------------*
       DO 270 L = 1, 10
          READ(10,*) TOSS
          READ(10,*) CAMT(CELL,L)
          READ(10,*) TOSS
          READ(10,*) CAMT(EXTR,L)
          READ(10,*) TOSS
          READ(10,*) CAMT(LIGN,L)
          READ(10,*) TOSS
          READ(10,*) CAMT(HUMU,L)
  270  CONTINUE

C      *-----------------------------------------------------*
C      * Read in the carbon to nitrogen ratio for cellolose, *
C      * extractives, lignin and humus.                      *
C      *-----------------------------------------------------*
       READ(10,*) TOSS
							READ(10,*) CTON(CELL)
       READ(10,*) TOSS
							READ(10,*) CTON(EXTR)
       READ(10,*) TOSS
							READ(10,*) CTON(LIGN)
       READ(10,*) TOSS
							READ(10,*) CTON(HUMU)

C      *----------------------------------*
C      * Read in the tranformation rates. *
C      *----------------------------------*
       READ(10,*) TOSS
							READ(10,*) TRANRATE(CE)
       READ(10,*) TOSS
							READ(10,*) TRANRATE(EL)
       READ(10,*) TOSS
							READ(10,*) TRANRATE(HE)
       READ(10,*) TOSS
							READ(10,*) TRANRATE(LE)
       READ(10,*) TOSS
							READ(10,*) TRANRATE(LH)
      
C      *-----------------------------------------*
C      * Read in the tranformation efficiencies. *
C      *-----------------------------------------*
       READ(10,*) TOSS
							READ(10,*) TRANEFFIC(CE)
       READ(10,*) TOSS
							READ(10,*) TRANEFFIC(EL)
       READ(10,*) TOSS
							READ(10,*) TRANEFFIC(HE)
       READ(10,*) TOSS
							READ(10,*) TRANEFFIC(LE)
       READ(10,*) TOSS
							READ(10,*) TRANEFFIC(LH)
      
C      *------------------------------------------------------*
C      * Read in the fraction of carbon and nitrogen in the   *
C      * dead roots for each layer for cellulose, extractives *
C      * and lignin.                                          *
C      *------------------------------------------------------*
       DO 280 L = 1, 10
          READ(10,*) TOSS
          READ(10,*) FRACTS(CELL, CARB, L)
          READ(10,*) TOSS
          READ(10,*) FRACTS(EXTR, CARB, L)
          READ(10,*) TOSS
          READ(10,*) FRACTS(LIGN, CARB, L)
  280  CONTINUE

C      *----------------------------------------------------*
C      * Read in the fraction of carbon and nitrogen in the *
C      * dead leaves.                                       *
C      *----------------------------------------------------*
       READ(10,*) TOSS
       READ(10,*) LFFRACTS(CELL,CARB)
       READ(10,*) TOSS
       READ(10,*) LFFRACTS(EXTR,CARB)
       READ(10,*) TOSS
       READ(10,*) LFFRACTS(LIGN,CARB)

C      *-------------------------------------------------------*
C      * Read in the ILC shielding parameter for cellulose and *
C      * extractives.                                          *
C      *-------------------------------------------------------*
       READ(10,*) TOSS
							READ(10,*) ILC(CELL)
       READ(10,*) TOSS
							READ(10,*) ILC(EXTR)

C      *---------------------------------------------------*
C      * Read in the maximal and optimal soil temperature. *
C      *---------------------------------------------------*
       READ(10,*) TOSS
       READ(10,*) MAXSLTEMP
       READ(10,*) TOSS
       READ(10,*) OPTSLTEMP

C      *-------------------------------*
C      * Read in the Q curve number.   *
C      *-------------------------------*
       READ(10,*) TOSS
       READ(10,*) QCURVE

C      *----------------------------------------------*
C      * Read in the decomposition N half saturation. *
C      *----------------------------------------------*
       READ(10,*) TOSS
       READ(10,*) HALFSAT

C      *----------------------------------------------------*
C      * Read in the optimal moisture, the moisture breadth *
C      * parameter and the moisture skewness parameter.     *
C      *----------------------------------------------------*
       READ(10,*) TOSS
       READ(10,*) OPTMOIST
       READ(10,*) TOSS
       READ(10,*) BRDTHMOIST
       READ(10,*) TOSS
       READ(10,*) SKEWMOIST

C      *-------------------------------*
C      * Read in the number of phases. *
C      *-------------------------------*
       READ(10,*) TOSS
       READ(10,*) NPHASES

C      *----------------------------------*
C      * Read in the names of the phases. *
C      *----------------------------------*
       DO 290 S = 1, 15
          READ(10,*) TOSS
          READ(10,*) PHASENAMES(S)
  290  CONTINUE

C      *-----------------------------------------------------*
C      * Read in the stoichiometric coefficients for phases. *
C      *-----------------------------------------------------*
       DO 300 C = 1, 15
          DO 310 S = 1, 15
             READ(10,*) TOSS
             READ(10,*) SP(S,C)
  310     CONTINUE
  300  CONTINUE

C      *-----------------------------------------------*
C      * Read in the equilibrium constants for phases. *
C      *-----------------------------------------------*
       DO 320 L = 1, 10
          DO 330 S = 1, 15
             READ(10,*) TOSS
             READ(10,*) SEC(S,L)
             SEC(S,L) = 10**SEC(S,L)
  330     CONTINUE
  320  CONTINUE

C      *-----------------------------------------------------*
C      * Read in the activity coefficient choices for phases.*
C      *-----------------------------------------------------*
       DO 340 S = 1, 15
          READ(10,*) TOSS
          READ(10,*) COMPACTCORR(S)
  340  CONTINUE
       DO 350 S = 1, 40
          READ(10,*) TOSS
          READ(10,*) SPECACTCORR(S)
  350  CONTINUE
         
C      *-----------------------------------------------------*
C      * Read in the mineralization rates for Ca, Mg and K.  *
C      *-----------------------------------------------------*
       READ(10,*) TOSS
       READ(10,*) CA_MIN_RAT
       READ(10,*) TOSS
       READ(10,*) MG_MIN_RAT
       READ(10,*) TOSS
       READ(10,*) K_MIN_RATE

C      *------------------------------------------------------*
C      * Read in the initial amounts of Ca, Mg, K, P and S in *
C      * organic matter.                                      *
C      *------------------------------------------------------*
       DO 360 L = 1, NLAYERSSIZE
          READ(10,*) TOSS
          READ(10,*) CA_IN_OM(L)
          READ(10,*) TOSS
          READ(10,*) MG_IN_OM(L)
          READ(10,*) TOSS
          READ(10,*) K_IN_OM(L)
          READ(10,*) TOSS
          READ(10,*) P_IN_OM(L)
          READ(10,*) TOSS
          READ(10,*) S_IN_OM(L)
  360  CONTINUE

C      *----------------------------------*
C      * Read in output information flag. *
C      *----------------------------------*
       READ(10,*) TOSS
       READ(10,*) OUTINFO

C      *------------------------------------------------------*
C      * Read in the file description at the end of the file. *
C      *------------------------------------------------------*
       READ(10,*) COMMENT

C      *----------------------*	   
C      * Output error message *
C      *----------------------*	
 9999  CONTINUE
       IF (GOAHEAD.EQ.0) THEN   
		        MSG(3) = ' Routine: InDat  '
	         WRITE(6,1000) (MSG(S), S=1,3)
	         WRITE(YASOUT,1000) (MSG(S), S=1,3)
 1000     FORMAT(' ',/,A80)
          PAUSE
       ENDIF

C      *----------------*
C      * Close the file *
C      *----------------*
       CLOSE(UNIT=10)

C      *---------------------------------------------------------*
C      * Transfer the labels, charges, and STypes into the their *
C      * proper arrays so that dependent species follow directly *
C      * after components.                                       *
C      *---------------------------------------------------------*
       DO 700 C = 1, 55
          LABELS(C) = 'none'
          HOLDSTYPE(C) = 'AqMassBal'
          Z(C) = 0
  700  CONTINUE
       DO 710 C = 1, NCOMP
          LABELS(C) = TEMPLABELS(C)
          HOLDSTYPE(C) = TEMPSTYPE(C)
          Z(C) = TEMPZ(C)
  710  CONTINUE
       S = 16
       DO 720 C = NCOMP+1, NSPECIES
          LABELS(C) = TEMPLABELS(S)
          HOLDSTYPE(C) = TEMPSTYPE(S)
          Z(C) = TEMPZ(S)
          S = S + 1
  720  CONTINUE

C      *--------------------------------------------------------*
C      * Calculate the initial amount of nitrogen in cellulose, *
C      * extractives, lignin and humus.                         *
C      *--------------------------------------------------------*
       DO 1100 L = 1, NLAYERS
          NAMT(CELL,L) = CAMT(CELL,L) / CTON(CELL)
          NAMT(EXTR,L) = CAMT(EXTR,L) / CTON(EXTR)
          NAMT(LIGN,L) = CAMT(LIGN,L) / CTON(LIGN)
          NAMT(HUMU,L) = CAMT(HUMU,L) / CTON(HUMU)
 1100  CONTINUE

C      *------------------------------------------------------*
C      * Transfer the activity coefficients into the activity *
C      * coefficient array so they are contiguous.            *
C      *------------------------------------------------------*
       L = 1
       DO 1110 S = 1, NCOMP
          HOLDAC(L) = COMPACTCORR(S)
          L = L + 1
 1110  CONTINUE
       DO 1120 S = 1, NDEPSP
          HOLDAC(L) = SPECACTCORR(S)
          L = L + 1
 1120  CONTINUE

C      *---------------------------------------------------------*
C      * Translate "word" species types to the number codes used *
C      * in the model.                                           *
C      *---------------------------------------------------------*
       DO 1130 S = 1, NSPECIES
          IF (HOLDSTYPE(S).EQ.'AqMassBal') THEN
             STYPE(S) = 1
          ELSE IF (HOLDSTYPE(S).EQ.'AqFixAct') THEN
             STYPE(S) = 2
          ELSE IF (HOLDSTYPE(S).EQ.'SPMassBal') THEN
             STYPE(S) = 11
          ELSE IF (HOLDSTYPE(S).EQ.'SPFixAct') THEN
             STYPE(S) = 12
          ELSE
             WRITE(6,*) 'ERROR ON INPUT OF STYPES.  TYPE DOES NOT EXIST'
             WRITE(6,*) 'SType = ', HOLDSTYPE(S)
             WRITE(YASOUT,*) 
     >       'ERROR ON INPUT OF STYPES.  TYPE DOES NOT EXIST'
             WRITE(YASOUT,*) 'SType = ', HOLDSTYPE(S)
             GOAHEAD = 0
          END IF
 1130  CONTINUE

C      *------------------------------------------------------*
C      * Translate "word" activity correction coefficients to *
C      * the number codes used in the model.                  *
C      *------------------------------------------------------*
       DO 1140 S = 1, NSPECIES
          IF (HOLDAC(S).EQ.'None') THEN
             ACTCORR(S) = 1
          ELSE IF (HOLDAC(S).EQ.'Davies') THEN
             ACTCORR(S) = 2
          ELSE IF (HOLDAC(S).EQ.'Vanse') THEN
             ACTCORR(S) = 4
          ELSE IF (HOLDAC(S).EQ.'Gain_Thom') THEN
             ACTCORR(S) = 5
          ELSE
             WRITE(6,*) 'ERROR ON INPUT OF ACTIVITY CORRECTION ',
     >       'COEFFICIENTS.  COEFFICIENT DOES NOT EXIST'
             WRITE(6,*) 'Activity correction coefficient = ', 
     >       HOLDAC(S)
             WRITE(YASOUT,*) 'ERROR ON INPUT OF ACTIVITY CORRECTION ',
     >       'COEFFICIENTS.  COEFFICIENT DOES NOT EXIST'
             WRITE(YASOUT,*) 'Activity correction coefficient = ', 
     >       HOLDAC(S)
             GOAHEAD = 0
          END IF
 1140  CONTINUE
	      RETURN
	      END
C      *****************************************************************
C      *                    END PROGRAM                                *
C      *****************************************************************
	   
